#!/bin/sh
# Start tclsh \
exec wish8.5 "$0" "$@"

# ------------------------------------------------------------
# test1.tcl
#
#   Example script testing event driven access of a K8055 board
#   using the k8055control utility.
# ------------------------------------------------------------


if {[catch {package require k8055m}]} {
    load tclk8055m.dll Tclk8055m
}
k8055m_init

proc main {argc argv} {

    frame   .all
    pack    .all

    foreach board {0 1 2 3} {
	button  .all.b$board    -borderwidth 2 -relief raised   \
				-text "Use K8055 #$board"       \
				-command [list setup_board $board]
	pack    .all.b$board    -side top
    }

    button      .all.refresh    -borderwidth 2 -relief raised   \
				-text "Refresh"                 \
				-command {refresh_boards}
    pack        .all.refresh    -side top -padx 4 -pady 4 -fill x

    button      .all.quit       -borderwidth 2 -relief raised   \
				-text "QUIT"                    \
				-command {destroy .}
    pack        .all.quit       -side top -padx 4 -pady 4 -fill x

    refresh_boards

    wm title . "K8055 demo2"
}


proc refresh_boards {} {
    foreach board {0 1 2 3} {
	if {[k8055m_card_present $board]} {
	    .all.b$board configure -state normal
	} else {
	    .all.b$board configure -state disabled
	}
    }
}


proc setup_board {board} {
    global	bdata${board}

    # ----
    # If the toplevel for this board exists, just raise it.
    # ----
    set top .board$board
    if {[winfo exists $top]} {
	wm deiconify $top
	raise $top
	return
    }

    # ----
    # Open the board and read relevant output data.
    # ----
    if {[catch {
	    k8055m_open $board

	    foreach d {0 1 2 3 4 5 6 7} e {O1 O2 O3 O4 O5 O6 O7 O8} {
	        set bdata${board}($e) [k8055m_get_digital_out $board $d]
	    }
	    set bdata${board}(DAC1) [k8055m_get_pwm_out $board 0]
	    set bdata${board}(DAC2) [k8055m_get_pwm_out $board 1]
	} err]} {
	tk_messageBox -type ok -message "$err"
	return
    }

    toplevel $top
    wm title $top "K8055 Board #$board"
    wm group $top .

    frame   $top.all        -borderwidth 0 -relief flat
    pack    $top.all        -side top -fill both

    grid rowconfigure $top.all 3 -weight 1

    # ----
    # The DAC sliders are very simple. Just a "scale" widget controlling
    # the DAC1 or DAC2 variable directly. We arrange them vertical with
    # a label at the top.
    # ----
    set dac $top.all.dac
    frame   $dac        -borderwidth 2 -relief sunken
    grid    $dac        -in $top.all -column 0 -row 0 -rowspan 4	\
			-sticky ns

    frame   $dac.dac1   -borderwidth 0 -relief flat
    pack    $dac.dac1   -side left
    frame   $dac.dac2   -borderwidth 0 -relief flat
    pack    $dac.dac2   -side left

    label   $dac.dac1.l     -borderwidth 2 -relief flat                 \
			    -text "DAC1" -anchor c
    pack    $dac.dac1.l     -side top -padx 2 -pady 2
    scale   $dac.dac1.s     -borderwidth 2 -relief sunken               \
			    -orient vertical -length 256                \
			    -from 1023 -to 0 -showvalue false           \
			    -variable bdata${board}(DAC1)		\
			    -command [list dac_change $board 0 DAC1]
    pack    $dac.dac1.s     -side top -padx 2 -pady 2
    label   $dac.dac1.v     -borderwidth 2 -relief sunken               \
			    -width 4 -anchor e                          \
			    -textvariable bdata${board}(DAC1)
    pack    $dac.dac1.v     -side top -padx 2 -pady 2

    label   $dac.dac2.l     -borderwidth 2 -relief flat                 \
			    -text "DAC2" -anchor c
    pack    $dac.dac2.l     -side top -padx 2 -pady 2
    scale   $dac.dac2.s     -borderwidth 2 -relief sunken               \
			    -orient vertical -length 256                \
			    -from 1023 -to 0 -showvalue false           \
			    -variable bdata${board}(DAC2)		\
			    -command [list dac_change $board 1 DAC2]
    pack    $dac.dac2.s     -side top -padx 2 -pady 2
    label   $dac.dac2.v     -borderwidth 2 -relief sunken               \
			    -width 4 -anchor e                          \
			    -textvariable bdata${board}(DAC2)
    pack    $dac.dac2.v     -side top -padx 2 -pady 2

    # ----
    # For the two analog inputs we need a canvas with a rectangle
    # inside that is adjusted by a call to analog_level
    # ----
    set ana $top.all.analog
    frame   $ana            -borderwidth 2 -relief sunken
    grid    $ana            -in $top.all -column 1 -row 0 -rowspan 4	\
			    -sticky ns

    frame   $ana.a1         -borderwidth 0 -relief flat
    pack    $ana.a1         -side left
    frame   $ana.a2         -borderwidth 0 -relief flat
    pack    $ana.a2         -side left

    label   $ana.a1.l       -borderwidth 2 -relief flat                 \
			    -text "A1" -anchor c
    pack    $ana.a1.l       -side top -padx 2 -pady 2
    canvas  $ana.a1.c       -borderwidth 2 -relief sunken               \
			    -highlightthickness 0 -bg white             \
			    -width 20 -height 256
    pack    $ana.a1.c       -side top -padx 2 -pady 2
    label   $ana.a1.v       -borderwidth 2 -relief sunken               \
			    -width 4 -anchor e                          \
			    -textvariable bdata${board}(A1)
    pack    $ana.a1.v       -side top -padx 2 -pady 2

    $ana.a1.c create rectangle 0 0 0 0  -outline {} -fill blue3     	\
			    -tags level

    label   $ana.a2.l       -borderwidth 2 -relief flat                 \
			    -text "A2" -anchor c
    pack    $ana.a2.l       -side top -padx 2 -pady 2
    canvas  $ana.a2.c       -borderwidth 2 -relief sunken               \
			    -highlightthickness 0 -bg white             \
			    -width 20 -height 256
    pack    $ana.a2.c       -side top -padx 2 -pady 2
    label   $ana.a2.v       -borderwidth 2 -relief sunken               \
			    -width 4 -anchor e                          \
			    -textvariable bdata${board}(A2)
    pack    $ana.a2.v       -side top -padx 2 -pady 2

    $ana.a2.c create rectangle 0 0 0 0  -outline {} -fill blue3     	\
			    -tags level

    # ----
    # Digital outputs are next.
    # ----
    set dig $top.all.digital
    frame   $dig            -borderwidth 2 -relief sunken
    grid    $dig            -in $top.all -column 2 -row 0 -sticky we

    for {set d 0} {$d < 8} {incr d} {
	label   $dig.l$d    -borderwidth 2 -relief flat			\
			    -text "D[expr $d + 1]" -anchor c
	grid    $dig.l$d    -in $dig -column $d -row 0			\
			    -padx 2 -pady 2
	checkbutton $dig.c$d -borderwidth 2 -relief flat		\
			    -variable bdata${board}(O[expr $d + 1])	\
			    -command [list digital_change $board $d O[expr $d + 1]]
	grid    $dig.c$d    -in $dig -column $d -row 1			\
			    -padx 4 -pady 4
    }

    # ----
    # The 5 digital input lines are small canvas items where
    # a callback function sets the background color to red or black.
    # ----
    set in $top.all.input
    frame   $in             -borderwidth 2 -relief sunken
    grid    $in             -in $top.all -column 2 -row 1 -sticky we

    for {set i 0} {$i < 5} {incr i} {
	grid columnconfigure $in $i -weight 1

	label   $in.l$i     -borderwidth 2 -relief flat			\
			    -text "I[expr $i + 1]" -anchor c
	grid    $in.l$i     -in $in -column $i -row 0			\
			    -padx 2 -pady 2
	canvas  $in.c$i     -borderwidth 2 -relief sunken		\
			    -width 15 -height 15 -bg black
	grid    $in.c$i     -in $in -column $i -row 1			\
			    -padx 4 -pady 4
    }

    # ----
    # The counters again just track the status variable but also
    # have a reset button. That button simply has the k8055_reset_counter
    # call as command.
    # ----
    set cntr $top.all.counter
    frame   $cntr           -borderwidth 2 -relief sunken
    grid    $cntr           -in $top.all -column 2 -row 2 -sticky we

    foreach c {0 1 2 3 4} {
	grid columnconfigure $cntr $c -weight 1

	label   $cntr.l$c   -borderwidth 2 -relief flat			\
			    -text "Counter [expr $c + 1]" -anchor c
	grid    $cntr.l$c   -in $cntr -column $c -row 0			\
			    -padx 2 -pady 2
    
	label   $cntr.v$c   -borderwidth 2 -relief sunken		\
			    -textvariable bdata${board}(C[expr $c + 1])	\
			    -width 8 -bg white -anchor e
	grid    $cntr.v$c   -in $cntr -column $c -row 1			\
			    -padx 4 -pady 4

	button  $cntr.r$c   -borderwidth 2 -relief raised		\
			    -text "Reset"				\
			    -command [list k8055m_set_counter $board $c 0]
	grid    $cntr.r$c   -in $cntr -column $c -row 2			\
			    -padx 2 -pady 2
    }

    # ----
    # Finally the debounce counter times. We use an entry with a
    # global variable and invoke a check/set function on the button.
    # ----
    set deb $top.all.debounce
    frame   $deb            -borderwidth 2 -relief sunken
    grid    $deb            -in $top.all -column 2 -row 3 -sticky nswe

    foreach c {0 1 2 3 4} {
	grid columnconfigure $deb $c -weight 1

	label   $deb.l$c    -borderwidth 2 -relief flat			\
			    -text "Debounce [expr $c + 1]" -anchor c
	grid    $deb.l$c    -in $deb -column $c -row 0			\
			    -padx 2 -pady 2
    
	entry   $deb.v$c    -borderwidth 2 -relief sunken		\
			    -textvariable bdata${board}(DEB[expr $c + 1]) \
			    -width 8 -bg white
	grid    $deb.v$c    -in $deb -column $c -row 1			\
			    -padx 4 -pady 4

	button  $deb.r$c    -borderwidth 2 -relief raised		\
			    -text "Set"					\
			    -command [list set_debounce_time $board $c DEB[expr $c + 1]]
	grid    $deb.r$c    -in $deb -column $c -row 2			\
			    -padx 2 -pady 2

	bind    $deb.v$c    <KeyPress-Return> [list $deb.r$c invoke]
    }

    # ----
    # Arrange for clean shutdown if the app is closed through
    # the window manager.
    # ----
    wm protocol $top WM_DELETE_WINDOW [list on_destroy $board $top]
    wm deiconify $top

    k8055m_set_report_interval $board 1 1000
    k8055m_register_callback $board [list input_callback $board]

    input_callback $board 1
}


proc input_callback {board {first_call 0}} {
    upvar #0	bdata${board}	bdata

    set all .board${board}.all

    foreach d {0 1 2 3 4} e {I1 I2 I3 I4 I5} c {C1 C2 C3 C4 C5}	deb {DEB1 DEB2 DEB3 DEB4 DEB5} {
    	set bdata($e) [k8055m_get_digital_in $board $d]
	set bdata($c) [k8055m_get_counter $board $d]

	if {$bdata($e)} {
	    $all.input.c${d} configure -bg red2 
	} else {
	    $all.input.c${d} configure -bg black 
	}

	if {$first_call} {
	    set bdata($deb) [k8055m_get_debounce_time $board $d]
	}
    }

    set bdata(A1) [k8055m_get_analog_in $board 0]
    set bdata(A2) [k8055m_get_analog_in $board 1]
    $all.analog.a1.c coords level [list 0 [expr (1023 - $bdata(A1)) / 4.0] 20 257]
    $all.analog.a2.c coords level [list 0 [expr (1023 - $bdata(A2)) / 4.0] 20 257]
}


proc dac_change {board port elem newval} {
    upvar #0	bdata${board}	bdata

    set bdata($elem) $newval
    k8055m_set_pwm_out $board $port $newval
}


proc digital_change {board port elem} {
    upvar #0	bdata${board}	bdata

    k8055m_set_digital_out $board $port $bdata($elem)
}


proc set_debounce_time {board c elem} {
    upvar #0	bdata${board}	bdata

    k8055m_set_debounce_time $board $c $bdata($elem)
}


proc on_destroy {board top} {
    catch {k8055m_close $board}
    destroy $top
    update
}


main $argc $argv
